/*
 * Created on 27.05.2004
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */
package networkconsole.ui;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.ResourceBundle;

import javax.xml.transform.sax.SAXSource;

import networkconsole.datamodel.JdbcEntryList;
import networkconsole.datamodel.LdapHostEntryList;
import networkconsole.datamodel.LdapUserEntryList;
import networkconsole.datamodel.swt.dialogs.ExportDialog;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.CoolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.ToolBarManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.window.ApplicationWindow;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;

/**
 * @author Eugene Prokopiev <eugene_prokopiev@mail.ru>
 * 
 */
public class NetworkConsoleWindow extends ApplicationWindow {

	static LdapHostEntryList hosts;
	static LdapUserEntryList users;
	static JdbcEntryList queryList; 
	static JdbcEntryList proceduresList; 
	
	public static Properties properties;	
	
	CTabFolder tabFolder;
	
	private CTabItem tabHostsView;
	private CTabItem tabUsersView;
	private CTabItem tabTrafficView;
	
	public NetworkConsoleWindow() {
		
		super(null);
		
		addMenuBar();
		addCoolBar(SWT.FLAT | SWT.WRAP);
		addStatusLine();
	}
	
	protected CoolBarManager createCoolBarManager(int style) {
		
		CoolBarManager coolBarManager = new CoolBarManager(style);
		coolBarManager.add(createToolBarManager(style));
		return coolBarManager;
	}
	
	protected ToolBarManager createToolBarManager(int style) {
		
		ToolBarManager toolBarManager = new ToolBarManager(style);
		
		toolBarManager.add(exitAction);
		toolBarManager.add(new Separator());
		toolBarManager.add(exportAction);
		toolBarManager.add(refreshAction);
		toolBarManager.add(new Separator());
		toolBarManager.add(addAction);
		toolBarManager.add(editAction);
		toolBarManager.add(removeAction);
		toolBarManager.add(queryAction);
		toolBarManager.add(new Separator());
		toolBarManager.add(aboutAction);
		
		return toolBarManager;
	}
	
	protected MenuManager createMenuManager() {
		
		MenuManager menuBar = new MenuManager();
		
		MenuManager fileMenu = new MenuManager(NetworkConsoleWindow.translate("ui.menu.file"));
		MenuManager editMenu = new MenuManager(NetworkConsoleWindow.translate("ui.menu.edit"));
		MenuManager helpMenu = new MenuManager(NetworkConsoleWindow.translate("ui.menu.help"));
		
		menuBar.add(fileMenu);
		menuBar.add(editMenu);
		menuBar.add(helpMenu);
		
		fileMenu.add(refreshAction);
		fileMenu.add(exportAction);
		fileMenu.add(new Separator());
		fileMenu.add(exitAction);
		
		editMenu.add(addAction);
		editMenu.add(editAction);
		editMenu.add(removeAction);
		editMenu.add(new Separator());
		editMenu.add(queryAction);
		
		helpMenu.add(aboutAction);
		
		return menuBar;
	}
	
	protected MenuManager createContextMenuManager() {
		
		MenuManager menuBar = new MenuManager();
		
		menuBar.add(addAction);
		menuBar.add(editAction);
		menuBar.add(removeAction);
		
		return menuBar;
	}
	
	protected Control createContents(Composite parent) {		
				
		Composite container = new Composite(parent, SWT.NONE);
		container.setLayout(new FillLayout());	
		
		tabFolder = new CTabFolder(container, SWT.BORDER | SWT.BOTTOM);
		tabFolder.setSelectionBackground(tabFolder.getBackground());
		tabFolder.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setActionsEnabled();
			}
		});
		
		tabUsersView = new NetworkUsersView(this, createContextMenuManager().createContextMenu(container));
		tabHostsView = new NetworkHostsView(this, createContextMenuManager().createContextMenu(container));
		tabTrafficView = new NetworkTrafficView(this);
		tabFolder.setFocus();
		
		return container;
	}
	
	protected void configureShell(Shell shell) {
		
		super.configureShell(shell);		
		shell.setText(NetworkConsoleWindow.translate("ui.caption"));
		shell.setImage(ImageDescriptor.createFromFile(getClass(), "images/console.gif").createImage());
	}
	
	public static void main(String[] args) throws IOException {
		
		Shell shell = new Shell();
		
		properties = new Properties();
		properties.load(new FileInputStream("networkconsole.properties"));	
		
		ConnectionDialog dialog = new ConnectionDialog(shell);
		if (dialog.open() == Window.OK) {
			NetworkConsoleWindow w = new NetworkConsoleWindow();
			w.setBlockOnOpen(true);				
			w.open();
		}
		
		NetworkConsoleWindow.saveProperties();	
		
		shell.getDisplay().dispose();
		shell.dispose();
	}
	
	public static String translate(String element) {
		return ResourceBundle.getBundle("networkconsole/ui/translation").getString(element);
	}

	public static void saveProperties() throws IOException {
		properties.store(new FileOutputStream("networkconsole.properties"), "Network Console Properties");
	}

	private RefreshAction refreshAction = new RefreshAction(this); 
	private ExportAction exportAction = new ExportAction(this); 
	private ExitAction exitAction = new ExitAction(this); 
	private AboutAction aboutAction = new AboutAction(this); 
	private AddAction addAction = new AddAction(this);
	private RemoveAction removeAction = new RemoveAction(this); 
	private EditAction editAction = new EditAction(this);
	private QueryAction queryAction = new QueryAction(this);

	public void setActionsEnabled() {
		addAction.setEnabled(((INetworkConsoleView)tabFolder.getSelection()).isAddEnabled());
		editAction.setEnabled(((INetworkConsoleView)tabFolder.getSelection()).isEditEnabled());
		removeAction.setEnabled(((INetworkConsoleView)tabFolder.getSelection()).isRemoveEnabled());
		queryAction.setEnabled(((INetworkConsoleView)tabFolder.getSelection()).isQueryEnabled());
	}
	
	private class ExitAction extends Action {
		NetworkConsoleWindow window;
		public ExitAction(NetworkConsoleWindow w) {
			window = w;	
			String actionName = NetworkConsoleWindow.translate("ui.actions.exit");
			setText(actionName+"@Alt+X");
			setToolTipText(actionName);  
			setImageDescriptor(ImageDescriptor.createFromFile(getClass(), "images/close.gif"));		
		}
		public void run() {
			System.exit(0);
		}
	}
	
	private class RefreshAction extends Action {
		NetworkConsoleWindow window;
		public RefreshAction(NetworkConsoleWindow w) {
			window = w;	
			String actionName = NetworkConsoleWindow.translate("ui.actions.refresh");
			setText(actionName+"@F5");
			setToolTipText(actionName);  
			setImageDescriptor(ImageDescriptor.createFromFile(getClass(), "images/refresh.gif"));
		}
		public void run() {
			((INetworkConsoleView)(tabFolder.getSelection())).refresh();			
		}
	}
	
	private class ExportAction extends Action {
		NetworkConsoleWindow window;
		public ExportAction(NetworkConsoleWindow w) {
			window = w;	
			String actionName = NetworkConsoleWindow.translate("ui.actions.export");
			setText(actionName+"@F6");
			setToolTipText(actionName);  
			setImageDescriptor(ImageDescriptor.createFromFile(getClass(), "images/export.gif"));
		}
		public void run() {
			SAXSource source = ((INetworkConsoleView)(tabFolder.getSelection())).export();	
			ExportDialog dialog = new ExportDialog(getShell(), source, properties);
			dialog.open();
			try {
				saveProperties();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
	
	private class AboutAction extends Action {
		NetworkConsoleWindow window;
		public AboutAction(NetworkConsoleWindow w) {
			window = w;	
			String actionName = NetworkConsoleWindow.translate("ui.actions.about");
			setText(actionName+"@Ctrl+?");
			setToolTipText(actionName);
			setImageDescriptor(ImageDescriptor.createFromFile(getClass(), "images/info.gif"));
		}
		public void run() {
			MessageDialog.openInformation(getShell(), getShell().getText(),
				NetworkConsoleWindow.translate("ui.actions.about.text"));					
		}
	}
	
	private class AddAction extends Action {
		NetworkConsoleWindow window;
		public AddAction(NetworkConsoleWindow w) {
			window = w;	
			String actionName = NetworkConsoleWindow.translate("ui.actions.add");
			setText(actionName+"@Ctrl+A");
			setToolTipText(actionName);
			setImageDescriptor(ImageDescriptor.createFromFile(getClass(), "images/add.gif"));
		}
		public void run() {	
			((INetworkConsoleView)(tabFolder.getSelection())).add();	
		}
	}
	
	private class RemoveAction extends Action {
		NetworkConsoleWindow window;
		public RemoveAction(NetworkConsoleWindow w) {
			window = w;	
			String actionName = NetworkConsoleWindow.translate("ui.actions.remove");
			setText(actionName+"@Ctrl+R");
			setToolTipText(actionName);
			setImageDescriptor(ImageDescriptor.createFromFile(getClass(), "images/remove.gif"));
		}
		public void run() {
			MessageDialog messageBox; 
			messageBox = new MessageDialog(getShell(), getShell().getText(), null, 
				NetworkConsoleWindow.translate("ui.actions.remove.warning"), 
				MessageDialog.WARNING, new String[] {
					NetworkConsoleWindow.translate("ui.dialog.yes"), 
					NetworkConsoleWindow.translate("ui.dialog.no")}, 
				1); 
			messageBox.open();
			if (messageBox.getReturnCode() == 0)
				((INetworkConsoleView)(tabFolder.getSelection())).remove();	
		}
	}
	
	private class EditAction extends Action {
		NetworkConsoleWindow window;
		public EditAction(NetworkConsoleWindow w) {
			window = w;	
			String actionName = NetworkConsoleWindow.translate("ui.actions.edit");
			setText(actionName+"@Ctrl+E");
			setToolTipText(actionName);
			setImageDescriptor(ImageDescriptor.createFromFile(getClass(), "images/edit.gif"));
		}
		public void run() {		
			((INetworkConsoleView)(tabFolder.getSelection())).edit();	
		}
	}
	
	private class QueryAction extends Action {
		NetworkConsoleWindow window;
		public QueryAction(NetworkConsoleWindow w) {
			window = w;	
			String actionName = NetworkConsoleWindow.translate("ui.actions.query");
			setText(actionName+"@F10");
			setToolTipText(actionName);
			setImageDescriptor(ImageDescriptor.createFromFile(getClass(), "images/query.gif"));
		}
		public void run() {
			((INetworkConsoleView)(tabFolder.getSelection())).query();	
		}
	}

}
